import { PathfindService } from '../pathfind.service';
import { Graph, GraphNode, GraphEdge } from '../navigation.entity';

describe('PathfindService', () => {
  let pathfindService: PathfindService;

  beforeEach(() => {
    pathfindService = new PathfindService();
  });

  const createNode = (
    id: string,
    label = '',
    x = 0,
    y = 0,
    z = 0,
    edges: GraphEdge[] = [],
  ): GraphNode => ({
    id,
    label,
    x,
    y,
    z,
    graphId: 'g1',
    edges,
  });

  const createEdge = (id: string, nodeAId: string, nodeBId: string): GraphEdge => ({
    id,
    nodeAId,
    nodeBId,
  });

  it('should return path between two directly connected nodes', () => {
    const edge = createEdge('e1', 'A', 'B');
    const nodeA = createNode('A', 'Node A', 0, 0, 0, [edge]);
    const nodeB = createNode('B', 'Node B', 1, 0, 0, [edge]);

    const graph: Graph = {
      id: 'g1',
      originX: 0,
      originY: 0,
      originZ: 0,
      scale: 1,
      nodes: [nodeA, nodeB],
    };

    const result = pathfindService.AStar(graph, nodeA, nodeB);
    expect(result.map(n => n.id)).toEqual(['A', 'B']);
  });

  it('should return shortest path between three connected nodes', () => {
    const edgeAB = createEdge('e1', 'A', 'B');
    const edgeBC = createEdge('e2', 'B', 'C');

    const nodeA = createNode('A', 'Node A', 0, 0, 0, [edgeAB]);
    const nodeB = createNode('B', 'Node B', 1, 0, 0, [edgeAB, edgeBC]);
    const nodeC = createNode('C', 'Node C', 2, 0, 0, [edgeBC]);

    const graph: Graph = {
      id: 'g1',
      originX: 0,
      originY: 0,
      originZ: 0,
      scale: 1,
      nodes: [nodeA, nodeB, nodeC],
    };

    const result = pathfindService.AStar(graph, nodeA, nodeC);
    expect(result.map(n => n.id)).toEqual(['A', 'B', 'C']);
  });

  it('should return empty array if no path exists', () => {
    const nodeA = createNode('A', 'Node A');
    const nodeB = createNode('B', 'Node B');

    const graph: Graph = {
      id: 'g1',
      originX: 0,
      originY: 0,
      originZ: 0,
      scale: 1,
      nodes: [nodeA, nodeB],
    };

    const result = pathfindService.AStar(graph, nodeA, nodeB);
    expect(result).toEqual([]);
  });

  it('should return diagonal path correctly using 3D heuristic', () => {
    const edge = createEdge('e1', 'A', 'B');
    const nodeA = createNode('A', 'Node A', 0, 0, 0, [edge]);
    const nodeB = createNode('B', 'Node B', 1, 1, 1, [edge]);

    const graph: Graph = {
      id: 'g1',
      originX: 0,
      originY: 0,
      originZ: 0,
      scale: 1,
      nodes: [nodeA, nodeB],
    };

    const result = pathfindService.AStar(graph, nodeA, nodeB);
    expect(result.map(n => n.id)).toEqual(['A', 'B']);
  });
});
